##################################
#
# Process MAAT output and plot
#
# Matt Craig
# Nov 2019
#
##################################

####################
#load and clean data
####################

#clear global env and load dependencies
rm(list=ls())
library(tidyverse)
library(cowplot)

#read model output data
#Michaelis-Menten model output
outMM <- read.csv('out_MM_highKm.csv')
#Reverse Michaelis-Menten model output
outRMM <- read.csv('out_RMM_highKm.csv')
#Linear (i.e. conventional) model output
outlin <- read.csv('out_lin.csv')

#define labels for model hypothesis combinations
model_levs <- c('Reference', 'Mineral saturation', 'Density-dependent growth', 'Density-dependent turnover', 'Density-dependent growth and turnover', 
                'DD_CUE & Min_sat', 'DD_turnover & Min_sat', 'DD_CUE & DD_turnover & Min_sat')

#function to process and plot data
process_data <- function(outfile){
  df1 <- outfile %>% 
    #renaming fields, grabbing steady state (200yr) values, and gathering c pools (i.e. wide to long format)
    rename(
      input = soil_decomp.litter,
      POM = cpools1,
      MBC = cpools2,
      MAOM = cpools3,
      MAOM_formation =soil_decomp.transfer.t2_to_3,
      POM_CUE=soil_decomp.transfer.t1_to_2,
      MAOM_CUE=soil_decomp.transfer.t3_to_2,
      MBturnover=soil_decomp.decomp.d2
    ) %>%
    filter(time == 200) %>%
    filter(POM_CUE == MAOM_CUE)%>%
    gather(c_pool, concentration, POM, MBC, MAOM) 
  
  MAOM_hyps <- levels(df1$MAOM_formation)
  CUE_hyps <- levels(df1$POM_CUE)
  MTR_hyps <- sort(levels(df1$MBturnover), TRUE)
  
  df1 <- df1 %>% mutate(model = case_when(
    MAOM_formation == MAOM_hyps[1] & POM_CUE == CUE_hyps[1] & MBturnover == MTR_hyps[1] ~ model_levs[1],
    MAOM_formation == MAOM_hyps[2] & POM_CUE == CUE_hyps[1] & MBturnover == MTR_hyps[1] ~ model_levs[2],
    MAOM_formation == MAOM_hyps[1] & POM_CUE == CUE_hyps[2] & MBturnover == MTR_hyps[1] ~ model_levs[3],
    MAOM_formation == MAOM_hyps[1] & POM_CUE == CUE_hyps[1] & MBturnover == MTR_hyps[2] ~ model_levs[4],
    MAOM_formation == MAOM_hyps[1] & POM_CUE == CUE_hyps[2] & MBturnover == MTR_hyps[2] ~ model_levs[5],
    MAOM_formation == MAOM_hyps[2] & POM_CUE == CUE_hyps[2] & MBturnover == MTR_hyps[1] ~ model_levs[6],
    MAOM_formation == MAOM_hyps[2] & POM_CUE == CUE_hyps[1] & MBturnover == MTR_hyps[2] ~ model_levs[7],
    MAOM_formation == MAOM_hyps[2] & POM_CUE == CUE_hyps[2] & MBturnover == MTR_hyps[2] ~ model_levs[8],
  )) 
  
  df1$model <- factor(df1$model, levels = model_levs[1:8])
  df1
}

#put the three datasets throught the data processing pipeline
MM <- process_data(outMM)
lin <- process_data(outlin)
RMM <- process_data(outRMM)
####################

################
#plotting output
#################
#define color-blind color pallette 
cbp1 <- c("#999999", "#E69F00", "#56B4E9", "#009E73",
          "#F0E442", "#0072B2", "#D55E00", "#CC79A7")


##########
#Figure3
#########
fig3top <- ggplot(lin %>%
             filter(model %in% c('Reference', 'Mineral saturation')), 
             aes(x=input, y= concentration, color = c_pool))+
  geom_point()+
  facet_wrap(~model, nrow = 1)+
  geom_line() +
  scale_color_manual(values = cbp1[c(2,6,4)])+
  labs(x = '', y = expression(SOC~(mg~C~g~soil^-1)), color = 'Pool')+
  theme_bw()+
  theme(panel.grid = element_blank(),
        legend.position = 'none'
        )

fig3bot <- ggplot(MM %>%
              filter(model %in% c('Reference', 'Mineral saturation')), 
            aes(x=input, y= concentration, color = c_pool))+
  geom_point()+
  facet_wrap(~model, nrow = 1)+
  geom_line() +
  scale_color_manual(values = cbp1[c(2,6,4)])+
  labs(x = expression(C~input~rate~(mg~C~g~soil^-1~yr^-1)), y = expression(SOC~(mg~C~g~soil^-1)), color = 'Pool')+
  lims(y = c(0,25))+
  theme_bw()+
  theme(panel.grid = element_blank(),
        legend.position = 'none',
        strip.background = element_blank(),
        strip.text = element_blank())

legend <- get_legend(ggplot(lin, aes(x=input, y=concentration, color = c_pool))+
                       geom_point()+geom_line()+theme_bw()+labs(color='SOC pool')+
                       scale_color_manual(values = cbp1[c(2,6,4)], 
                                          labels = c(expression(C[m]), expression(C[b]), expression(C[p]))))
blank <- ggplot()+theme_classic()+theme(axis.line=element_blank())

make_mbc_inset <- function(mod){
  ggplot(lin %>%
           filter(c_pool == 'MBC',
                  model == mod),
         aes(x=input, y=concentration))+
    geom_point(color = cbp1[6])+
    geom_line(color = cbp1[6])+
    scale_y_continuous(breaks = c(1,3,5), limits = c(0,5))+
    theme_bw()+
    theme(panel.grid = element_blank(),
          legend.position = 'none',
          axis.text.x = element_blank(),
          axis.title.y = element_blank(),
          axis.title.x = element_blank())
}

insetA <- make_mbc_inset('Reference')
insetB <- make_mbc_inset('Mineral saturation')

figure3 <- 
  plot_grid(blank, plot_grid(plot_grid(fig3top,fig3bot, nrow = 2, align = 'v'), legend, ncol = 2, rel_widths = c(.85,.15)), nrow = 2, rel_heights = c(.04,.96))+  
  draw_label('(c)', 0.155, .44, fontface = 'bold') +
  draw_label('(d)', 0.54, .44, fontface = 'bold') +
  draw_label('Conventional', 0.46, .98, fontface = 'bold.italic') +
  draw_label('Microbial', 0.46, .5, fontface = 'bold.italic') +
  draw_plot(insetA, 0.097, .734, .17, .17 ) +
  draw_label('(a)', 0.155, .86, fontface = 'bold') +
  draw_plot(insetB, .48, .734, .17, .17) +
  draw_label('(b)', 0.54, .86, fontface = 'bold')
############################################################

##########
#Figure4
#########
fig4top <- ggplot(MM %>% filter(c_pool %in% c('POM', 'MAOM'),
  model %in% c('Density-dependent growth', 'Density-dependent turnover', 'Density-dependent growth and turnover')), 
  aes(x=input, y= concentration, color = c_pool))+
  geom_point()+
  facet_wrap(~model, nrow = 1)+
  geom_line() +
  scale_color_manual(values = cbp1[c(2,4)])+
  labs(x = '', y = expression(SOC~(mg~C~g~soil^-1)), color = 'Pool')+
  lims(y = c(0,25))+
  theme_bw()+
  theme(panel.grid = element_blank(),
        legend.position = 'none'
        )

fig4bot <- ggplot(MM %>% filter(c_pool %in% c('MBC'),
                     model %in% c('Density-dependent growth', 'Density-dependent turnover', 'Density-dependent growth and turnover')), 
       aes(x=input, y= concentration, color = c_pool))+
  geom_point(color = cbp1[6])+
  facet_wrap(~model, nrow = 1)+
  geom_line(color = cbp1[6]) +
  labs(x = expression(C~input~rate~(mg~C~g~soil^-1~yr^-1)), y = expression(C[b]~(mg~C~g~soil^-1)))+
  theme_bw()+
  theme(panel.grid = element_blank(),
        legend.position = 'none',
        strip.background = element_blank(),
        strip.text = element_blank()
  )

figure4 <- plot_grid(plot_grid(fig4top,fig4bot, nrow = 2, align = 'v'), legend, ncol = 2, rel_widths = c(.9,.1))+
  draw_label('(a)', 0.08, .93, fontface = 'bold') +
  draw_label('(b)', 0.36, .93, fontface = 'bold') +
  draw_label('(c)', 0.64, .93, fontface = 'bold') +
  draw_label('(d)', 0.08, .46, fontface = 'bold') +
  draw_label('(e)', 0.36, .46, fontface = 'bold') +
  draw_label('(f)', 0.64, .46, fontface = 'bold') 

############################################################

#########################
#output and name each plot
#########################
# tiff('plots/figure3.tiff', res = 300, width = 16.5, height = 15, units = 'cm')
# figure3
# dev.off()
# 
# tiff('plots/figure4.tiff', res = 300, width = 24, height = 16.5, units = 'cm')
# figure4
# dev.off()

#########################


########
#Reverse Michaelis-Menten figures for supplement
########

##########
#Figure S3
#########

figSupp3 <- ggplot(RMM %>%
                     filter(model %in% c('Reference', 'Mineral saturation')), 
                   aes(x=input, y= concentration, color = c_pool))+
  geom_point()+
  facet_wrap(~model, nrow = 1)+
  geom_line() +
  scale_color_manual(values = cbp1[c(2,6,4)])+
  labs(x = expression(C~input~rate~(mg~C~g~soil^-1~yr^-1)), y = expression(SOC~(mg~C~g~soil^-1)), color = 'Pool')+
  lims(y = c(0,25))+
  theme_bw()+
  theme(panel.grid = element_blank(),
        legend.position = 'none'
  )

legend <- get_legend(ggplot(lin, aes(x=input, y=concentration, color = c_pool))+
                       geom_point()+geom_line()+theme_bw()+labs(color='SOC pool')+
                       scale_color_manual(values = cbp1[c(2,6,4)], 
                                          labels = c(expression(C[m]), expression(C[b]), expression(C[p]))))
blank <- ggplot()+theme_classic()+theme(axis.line=element_blank())

figureSupp3 <- plot_grid(figSupp3,legend, ncol = 2, rel_widths = c(.85,.15)) + 
  draw_label('A', 0.13, 0.85, fontface = 'bold') +
  draw_label('B', 0.52, 0.85, fontface = 'bold')

############################################################

##########
#Figure S4
#########
figSupp4top <- ggplot(RMM %>% filter(c_pool %in% c('POM', 'MAOM'),
                                model %in% c('Density-dependent growth', 'Density-dependent turnover', 'Density-dependent growth and turnover')), 
                  aes(x=input, y= concentration, color = c_pool))+
  geom_point()+
  facet_wrap(~model, nrow = 1)+
  geom_line() +
  scale_color_manual(values = cbp1[c(2,4)])+
  labs(x = '', y = expression(SOC~(mg~C~g~soil^-1)), color = 'Pool')+
  lims(y = c(0,30))+
  theme_bw()+
  theme(panel.grid = element_blank(),
        legend.position = 'none'
  )

figSupp4bot <- ggplot(RMM %>% filter(c_pool %in% c('MBC'),
                                model %in% c('Density-dependent growth', 'Density-dependent turnover', 'Density-dependent growth and turnover')), 
                  aes(x=input, y= concentration, color = c_pool))+
  geom_point(color = cbp1[6])+
  facet_wrap(~model, nrow = 1)+
  geom_line(color = cbp1[6]) +
  labs(x = expression(C~input~rate~(mg~C~g~soil^-1~yr^-1)), y = expression(C[b]~(mg~C~g~soil^-1)))+
  theme_bw()+
  theme(panel.grid = element_blank(),
        legend.position = 'none',
        strip.background = element_blank(),
        strip.text = element_blank()
  )

figureSupp4 <- plot_grid(plot_grid(figSupp4top,figSupp4bot, nrow = 2, align = 'v'), legend, ncol = 2, rel_widths = c(.9,.1))+
  draw_label('A', 0.08, .93, fontface = 'bold') +
  draw_label('B', 0.36, .93, fontface = 'bold') +
  draw_label('C', 0.64, .93, fontface = 'bold') +
  draw_label('D', 0.08, .46, fontface = 'bold') +
  draw_label('E', 0.36, .46, fontface = 'bold') +
  draw_label('F', 0.64, .46, fontface = 'bold') 

############################################################


 # tiff('plots/figureSupp3.tiff', res = 100, width = 16.5, height = 8, units = 'cm')
 # figureSupp3
 # dev.off()
 # 
 # tiff('plots/figureSupp4.tiff', res = 100, width = 24, height = 16.5, units = 'cm')
 # figureSupp4
 # dev.off()
